//=====================================================================================================================
// Copyright (c) 2016-2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
//=====================================================================================================================

package com.actional.soapapi;

import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;

import javax.xml.rpc.ServiceException;

import com.actional.config.BaseData;
import com.actional.soapstation.task.IConfigTask;
import com.actional.soapstation.task.IConfigTaskServiceLocator;
import com.actional.soapstation.task.IProductInfoTask;
import com.actional.soapstation.task.IProductInfoTaskServiceLocator;

/** <!-- ========================================================================================================== -->
* This class provides typical services exposed all products.
* <!-- --------------------------------------------------------------------------------------------------------- --> */

public class Config
{
	/** <!-- ================================================================================================== -->
	* The Locators for the task services
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected IConfigTaskServiceLocator		configTaskLocator = new IConfigTaskServiceLocator()
								{
									public javax.xml.rpc.Call createCall()
										throws ServiceException
									{
										return prepareCall(super.createCall());
									}
								};
	protected IProductInfoTaskServiceLocator	productInfoTaskLocator = new IProductInfoTaskServiceLocator()
								{
									public javax.xml.rpc.Call createCall()
										throws ServiceException
									{
										return prepareCall(super.createCall());
									}
								};

	/** <!-- ================================================================================================== -->
	* The Tasks
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public IConfigTask	configTask;
	public IProductInfoTask	productInfoTask;

	/** <!-- ================================================================================================== -->
	 * Information about the user connecting.
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	protected String	itsUserName;
	protected String	itsPassword;
	protected String	itsEndpoint;
	protected String	itsImpersonateCookie;

	/** This should be set if we want axis to maintain the session between calls to the same stub. Axis does
	* not appear to share the session across stubs. */
	protected boolean	itsMaintainSession;

	/** <!-- ================================================================================================== -->
	 * Constructor
	 * <!-- ------------------------------------------------------------------------------------------------- --> */

	public Config(String userName, String password, String endpoint)
	{
		itsUserName = userName;
		itsPassword = password;
		itsEndpoint = endpoint;
	}

	public boolean getMaintainSession()
	{
		return itsMaintainSession;
	}

	/** <!-- ================================================================================================== -->
	* This method should be called before {@link #init()}. It turns on the maintaining
	* of session in the client stubs.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public void setMaintainSession(boolean val)
	{
		itsMaintainSession = val;
	}

	/** <!-- ================================================================================================== -->
	* This method should be called before {@link #init()}. It sets a cookie in the client stubs containing
	* the tired admin credentials to use w
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public void setImpersonate(String userName, String role, String group)
	{
		final StringBuilder sb = new StringBuilder(128);
		boolean hasInfo = false;

		sb.append("Actional_Impersonate=");

		if (!Util.isBlank(userName))
		{
			hasInfo = true;
			sb.append(userName);
		}

		if (!Util.isBlank(role) || !Util.isBlank(group))
		{
			hasInfo = true;
			sb.append('/');

			if (!Util.isBlank(role))
				sb.append(role);

			if (!Util.isBlank(group))
			{
				sb.append('/');
				sb.append(group);
			}
		}

		if (hasInfo)
			itsImpersonateCookie = sb.toString();
		else
			itsImpersonateCookie = null;
	}

	/** <!-- ================================================================================================== -->
	* Clients call this method to initialize the SDK.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public void init() throws Exception
	{
		try
		{
			prepareTasks();
		}
		catch (Exception e)
		{
			throw new Exception("Failed to initialize SOAP API connection.", e);
		}
	}

	/** <!-- ================================================================================================== -->
	* Initialize the axis stubs.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	protected void prepareTasks()
		throws ServiceException, MalformedURLException
	{
		configTask	= configTaskLocator.getIConfigTask(		proxyUrl(configTaskLocator.getIConfigTaskAddress()) );
		productInfoTask	= productInfoTaskLocator.getIProductInfoTask(	proxyUrl(productInfoTaskLocator.getIProductInfoTaskAddress()) );

		prepareStub(configTask);
		prepareStub(productInfoTask);
	}

	protected void prepareStub(Object o)
	{
		org.apache.axis.client.Stub stub = (org.apache.axis.client.Stub)o;

		// Set the credentials to use when talking to the server
		stub.setUsername(itsUserName);
		stub.setPassword(itsPassword);
		stub.setMaintainSession(itsMaintainSession);

		// Turn off the sending of the xsi:Type attribute
		stub._setProperty(org.apache.axis.client.Call.SEND_TYPE_ATTR, Boolean.FALSE);
	}

	protected URL proxyUrl(String url) throws MalformedURLException
	{
		String newUrl = url;

		if (itsEndpoint != null)
		{
			int pos = url.lastIndexOf("/api");

			if (pos > 0)
			{
				newUrl = url.substring(pos);
				newUrl = itsEndpoint + newUrl;
			}
		}

		return new URL(newUrl);
	}

	public javax.xml.rpc.Call prepareCall(javax.xml.rpc.Call rpcCall) throws ServiceException
	{
		org.apache.axis.client.Call axisCall = (org.apache.axis.client.Call) rpcCall;

		if (axisCall == null)
			return null;

		if (!Util.isBlank(itsImpersonateCookie))
		{
			axisCall.setScopedProperty(
					org.apache.axis.transport.http.HTTPConstants.HEADER_COOKIE,
					itsImpersonateCookie);
			axisCall.setScopedProperty(
					org.apache.axis.transport.http.HTTPConstants.HEADER_COOKIE2,
					itsImpersonateCookie);
		}

		return axisCall;
	}

	/** <!-- ================================================================================================== -->
	* This function will return an non-leaf version of the BaseData with the specified KeyID.<br>
	* <br>
	* When a class that extends BaseData is returned from the product there is no guarantie
	* that it will not be a {@link Util#isLeaf(BaseData) Leaf}. Leaf BaseData's only have the
	* {@link BaseData#getKeyID()}, {@link BaseData#getKeyName()}, {@link BaseData#getCode()}
	* fields set. All other fields in the class are null. To get the values of the other
	* fields, the BaseData has to be <i>acervated</i>. This is done by calling {@link #acervate(String)}.
	* <br>
	* <br>
	* <b>Definition:</b> Heaped, or growing in heaps, or closely compacted clusters.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public BaseData acervate(String keyID) throws RemoteException
	{
		return configTask.get(keyID);
	}

	/** This will acervate the BaseData if it is a leaf (returns new instance) otherwise
	* it returns the current instance. */
	public BaseData acervate(BaseData bd) throws RemoteException
	{
		if (bd == null)
			return null;

		if (Util.isLeaf(bd))
			return acervate(bd.getKeyID());
		else
			return bd;
	}

	/** This will check all the BaseData's in the array and acervate any that are leafs. */
	public void acervate(BaseData[] source) throws RemoteException
	{
		if (source == null)
			return;

		int len = source.length;

		for (int i = 0; i < len; i++)
			source[i] = acervate(source[i]);

		return;
	}

	/** <!-- ================================================================================================== -->
	* This method queries the config manager to make sure that the keyID exists and then creates a new
	* Leaf of the BaseData.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public BaseData _getLeaf(String keyID) throws Exception
	{
		// This method calls get on the config manager to ensure
		// that the object exists and then returns a leaf
		// version of the object.
		return Util.createLeaf(configTask.get(keyID));
	}

	/** <!-- ================================================================================================== -->
	* Find operations for performing QBEs on the Config Manager.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public BaseData[] find(Class<? extends BaseData> clazz)  throws Exception
	{
		return find(clazz, null);
	}

	public BaseData[] find(Class<? extends BaseData> clazz, String keyName)  throws Exception
	{
		final BaseData newBD = clazz.newInstance();

		newBD.setKeyName(keyName);

		return configTask.find(newBD);
	}

	public BaseData findFirst(Class<? extends BaseData> clazz, String keyName) throws Exception
	{
		BaseData res = findFirstIfExists(clazz, keyName);

		if (Util.isBlank(res))
		{
			throw new Exception("Failed to locate classes of type '" + clazz.getName() +
						"' with a KeyName of '" + keyName + "'");
		}

		return res;
	}

	public BaseData findFirstIfExists(Class<? extends BaseData> clazz, String keyName) throws Exception
	{
		BaseData[] res = find(clazz, keyName);

		if (Util.isBlank(res))
			return null;

		return res[0];
	}

	public BaseData findFirst(BaseData bd) throws Exception
	{
		BaseData res = findFirstIfExists(bd);

		if (Util.isBlank(res))
			throw new Exception("Failed to locate BaseData with criteria: " + bd);

		return res;
	}

	public BaseData findFirstIfExists(BaseData bd) throws Exception
	{
		BaseData[] res = configTask.find(bd);

		if (Util.isBlank(res))
			return null;

		return res[0];
	}

	/** <!-- ================================================================================================== -->
	* Find operations to follow the hidden back references in a class
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public BaseData[] findHiddenUsers(String keyID) throws Exception
	{
		return configTask.findHiddenUsers(keyID);
	}

	public BaseData[] findHiddenUsers(BaseData bd) throws Exception
	{
		return findHiddenUsers(bd.getKeyID());
	}

	public BaseData findFirstHiddenUser(String keyID) throws Exception
	{
		BaseData[] res = findHiddenUsers(keyID);

		if (Util.isBlank(res))
			throw new Exception("Failed to locate a user of the class with a keyID of: " + keyID);

		return res[0];
	}

	public BaseData findFirstHiddenUser(BaseData bd) throws Exception
	{
		return findFirstHiddenUser(bd.getKeyID());
	}

	/** <!-- ================================================================================================== -->
	* These methods check the permissions on the specified BaseData instance. This is done by sending
	* the BaseData to the server. We only need to send a leaf version of the BaseData which is much smaller.
	* <!-- ------------------------------------------------------------------------------------------------- --> */

	public boolean canRead(BaseData bd) throws Exception
	{
		if (bd == null)
			return false;

		return configTask.canRead(Util.createLeaf(bd), null, -1);
	}

	public boolean canWrite(BaseData bd) throws Exception
	{
		if (bd == null)
			return false;

		return configTask.canWrite(Util.createLeaf(bd), null, -1);
	}

	public boolean canCreate(BaseData bd) throws Exception
	{
		if (bd == null)
			return false;

		return configTask.canCreate(Util.createLeaf(bd), null, -1);
	}

	public boolean canDelete(BaseData bd) throws Exception
	{
		if (bd == null)
			return false;

		return configTask.canDelete(Util.createLeaf(bd), null, -1);
	}
}
